<?php
require_once __DIR__ . '/config/init.php';


// Verify state parameter to prevent CSRF
if (!isset($_GET['state']) || $_GET['state'] !== $_SESSION['oauth_state']) {
    die('Invalid state parameter. Possible CSRF attack.');
}

// Check for authorization code
if (isset($_GET['code'])) {
    try {
        // Create Google Client
        $client = createGoogleClient();
        
        // Exchange authorization code for access token
        $token = $client->fetchAccessTokenWithAuthCode($_GET['code']);
        
        if (isset($token['error'])) {
            throw new Exception('Error fetching access token: ' . $token['error_description']);
        }
        
        // Set the access token
        $client->setAccessToken($token);
        
        // Get user's Google Analytics accounts and properties
        $analyticsAdmin = new Google_Service_GoogleAnalyticsAdmin($client);
        
        $properties = [];
        
        try {
            // First, get all accounts
            $accounts = $analyticsAdmin->accounts->listAccounts();
            
            foreach ($accounts->getAccounts() as $account) {
                $accountName = $account->getName(); // Format: accounts/123456
                
                try {
                    // Get properties for this account
                    $accountProperties = $analyticsAdmin->properties->listProperties([
                        'filter' => 'parent:' . $accountName
                    ]);
                    
                    foreach ($accountProperties->getProperties() as $property) {
                        $properties[] = [
                            'id' => $property->getName(), // Format: properties/123456789
                            'name' => $property->getDisplayName(),
                            'type' => 'GA4',
                            'propertyId' => str_replace('properties/', '', $property->getName()),
                            'accountId' => str_replace('accounts/', '', $accountName),
                            'accountName' => $account->getDisplayName()
                        ];
                    }
                } catch (Exception $e) {
                    error_log('Error fetching properties for account ' . $accountName . ': ' . $e->getMessage());
                    continue; // Skip this account and continue with others
                }
            }
            
        } catch (Exception $e) {
            error_log('Error fetching accounts: ' . $e->getMessage());
            
            // Fallback: Try to get properties without account filtering
            try {
                $allProperties = $analyticsAdmin->properties->listProperties();
                
                foreach ($allProperties->getProperties() as $property) {
                    $properties[] = [
                        'id' => $property->getName(),
                        'name' => $property->getDisplayName(),
                        'type' => 'GA4',
                        'propertyId' => str_replace('properties/', '', $property->getName()),
                        'accountId' => '',
                        'accountName' => 'Unknown Account'
                    ];
                }
            } catch (Exception $e2) {
                error_log('Fallback method also failed: ' . $e2->getMessage());
                // If both methods fail, we'll show an empty properties list
            }
        }
        
        // Store data in session
        $_SESSION['ga_access_token'] = $token['access_token'];
        $_SESSION['ga_refresh_token'] = $token['refresh_token'] ?? null;
        $_SESSION['ga_token_expires'] = time() + $token['expires_in'];
        $_SESSION['ga_properties'] = $properties;
        $_SESSION['ga_full_token'] = $token;
        
        // Debug output
        echo "<div style='font-family: Arial, sans-serif; max-width: 800px; margin: 20px auto; padding: 20px;'>";
        echo "<h2>Google Analytics Connection Successful</h2>";
        
        if (empty($properties)) {
            echo "<div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
            echo "<h3>⚠️ No Properties Found</h3>";
            echo "<p>We couldn't find any Google Analytics 4 properties associated with your account. This could be because:</p>";
            echo "<ul>";
            echo "<li>You don't have any GA4 properties set up</li>";
            echo "<li>You don't have the necessary permissions to view the properties</li>";
            echo "<li>The properties are in a different Google account</li>";
            echo "</ul>";
            echo "<p><strong>What to do:</strong></p>";
            echo "<ol>";
            echo "<li>Make sure you have GA4 properties set up in your Google Analytics account</li>";
            echo "<li>Ensure you have at least 'Viewer' access to the properties</li>";
            echo "<li>Try connecting with a different Google account if needed</li>";
            echo "</ol>";
            echo "</div>";
        } else {
            echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
            echo "<h3>✅ Found " . count($properties) . " GA4 Properties</h3>";
            echo "</div>";
            
            echo "<h3>Available Properties:</h3>";
            echo "<table style='width: 100%; border-collapse: collapse; margin: 20px 0;'>";
            echo "<thead>";
            echo "<tr style='background: #f8f9fa;'>";
            echo "<th style='border: 1px solid #dee2e6; padding: 12px; text-align: left;'>Property Name</th>";
            echo "<th style='border: 1px solid #dee2e6; padding: 12px; text-align: left;'>Property ID</th>";
            echo "<th style='border: 1px solid #dee2e6; padding: 12px; text-align: left;'>Account</th>";
            echo "<th style='border: 1px solid #dee2e6; padding: 12px; text-align: left;'>Type</th>";
            echo "</tr>";
            echo "</thead>";
            echo "<tbody>";
            
            foreach ($properties as $property) {
                echo "<tr>";
                echo "<td style='border: 1px solid #dee2e6; padding: 12px;'>" . htmlspecialchars($property['name']) . "</td>";
                echo "<td style='border: 1px solid #dee2e6; padding: 12px;'>" . htmlspecialchars($property['propertyId']) . "</td>";
                echo "<td style='border: 1px solid #dee2e6; padding: 12px;'>" . htmlspecialchars($property['accountName']) . "</td>";
                echo "<td style='border: 1px solid #dee2e6; padding: 12px;'><span style='background: #007bff; color: white; padding: 2px 8px; border-radius: 3px; font-size: 12px;'>" . htmlspecialchars($property['type']) . "</span></td>";
                echo "</tr>";
            }
            
            echo "</tbody>";
            echo "</table>";
        }
        
        echo "<div style='margin: 30px 0;'>";
        echo "<a href='".$pluginURL."&route=select-property' style='background: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block;'>Continue to Select Property</a>";
        echo " ";
        echo "<a href='".$pluginURL."&route=connect' style='background: #6c757d; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block;'>Try Again</a>";
        echo "</div>";
        
        echo "</div>";
        
        exit;
        
    } catch (Exception $e) {
        echo "<div style='font-family: Arial, sans-serif; max-width: 800px; margin: 20px auto; padding: 20px;'>";
        echo "<h2 style='color: #dc3545;'>❌ Authentication Error</h2>";
        echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        echo "<p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
        echo "</div>";
        
        echo "<h3>Troubleshooting Steps:</h3>";
        echo "<ol>";
        echo "<li><strong>Check your Google Cloud Console setup:</strong>";
        echo "<ul>";
        echo "<li>Make sure the Google Analytics Admin API is enabled</li>";
        echo "<li>Verify your OAuth 2.0 credentials are correct</li>";
        echo "<li>Check that the redirect URI matches exactly</li>";
        echo "</ul>";
        echo "</li>";
        echo "<li><strong>Verify your Google Analytics access:</strong>";
        echo "<ul>";
        echo "<li>Make sure you have access to Google Analytics 4 properties</li>";
        echo "<li>Ensure you're using the correct Google account</li>";
        echo "</ul>";
        echo "</li>";
        echo "<li><strong>Check the OAuth scopes:</strong>";
        echo "<ul>";
        echo "<li>Make sure you're requesting the correct scopes</li>";
        echo "<li>Try clearing your browser cache and cookies</li>";
        echo "</ul>";
        echo "</li>";
        echo "</ol>";
        
        echo "<a href='".$pluginURL."&route=connect' style='background: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; margin-top: 20px;'>Try Again</a>";
        echo "</div>";
    }
} else {
    echo "<div style='font-family: Arial, sans-serif; max-width: 800px; margin: 20px auto; padding: 20px;'>";
    echo "<h2 style='color: #dc3545;'>❌ Authorization Failed</h2>";
    echo "<p>Authorization code not received.</p>";
    
    if (isset($_GET['error'])) {
        echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        echo "<p><strong>Error:</strong> " . htmlspecialchars($_GET['error']) . "</p>";
        if (isset($_GET['error_description'])) {
            echo "<p><strong>Description:</strong> " . htmlspecialchars($_GET['error_description']) . "</p>";
        }
        echo "</div>";
    }
    
    echo "<a href='".$pluginURL."&route=connect' style='background: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block;'>Try Again</a>";
    echo "</div>";
}
?>