<?php

class GoogleAnalyticsService {
    private $client;
    private $analyticsData;
    private $analytics;
    private $propertyType;
    private $propertyId;
    private $viewId;
    
    public function __construct($credentials) {
        $this->client = createGoogleClient();
        
        // Set access token
        $fullToken = json_decode($credentials['full_token'], true);
        $this->client->setAccessToken($fullToken);
        
        // Refresh token if needed
        if ($this->client->isAccessTokenExpired()) {
            if ($credentials['refresh_token']) {
                $this->client->refreshToken($credentials['refresh_token']);
                $newToken = $this->client->getAccessToken();
                
                // Update token in database
                $this->updateTokenInDatabase($credentials['admin_id'], $newToken);
            } else {
                throw new Exception('Access token expired and no refresh token available');
            }
        }
        
        $this->propertyType = $credentials['property_type'];
        $this->propertyId = $credentials['property_id'];
        $this->viewId = $credentials['view_id'];
        
        // Initialize appropriate service
        if ($this->propertyType === 'GA4') {
            $this->analyticsData = new Google_Service_AnalyticsData($this->client);
        } else {
            $this->analytics = new Google_Service_Analytics($this->client);
        }
    }
    
    private function updateTokenInDatabase($userId, $newToken) {
        global $db;
        
        $stmt = $db->table('google_analytics_credentials')->where('admin_id', $userId);
        
        $stmt->update([
            'access_token' => $newToken['access_token'],
            'token_expires' => time() + $newToken['expires_in'],
            'full_token' => json_encode($newToken)
            ]);
        
    }
    
    public function getOverviewMetrics($startDate, $endDate) {
        if ($this->propertyType === 'GA4') {
            return $this->getGA4OverviewMetrics($startDate, $endDate);
        } else {
            return $this->getUAOverviewMetrics($startDate, $endDate);
        }
    }
    
    private function getGA4OverviewMetrics($startDate, $endDate) {
        try {
            $request = new Google_Service_AnalyticsData_RunReportRequest();
            $request->setProperty('properties/' . $this->propertyId);
            
            // Set date range
            $dateRange = new Google_Service_AnalyticsData_DateRange();
            $dateRange->setStartDate($startDate);
            $dateRange->setEndDate($endDate);
            $request->setDateRanges([$dateRange]);
            
            // Set metrics
            $metrics = [
                $this->createMetric('totalUsers'),
                $this->createMetric('sessions'),
                $this->createMetric('screenPageViews'),
                $this->createMetric('averageSessionDuration')
            ];
            $request->setMetrics($metrics);
            
            $response = $this->analyticsData->properties->runReport('properties/' . $this->propertyId, $request);
            
            $result = [
                'users' => 0,
                'sessions' => 0,
                'pageviews' => 0,
                'avgSessionDuration' => 0
            ];
            
            if ($response->getRows()) {
                $row = $response->getRows()[0];
                $metricValues = $row->getMetricValues();
                
                $result['users'] = (int)$metricValues[0]->getValue();
                $result['sessions'] = (int)$metricValues[1]->getValue();
                $result['pageviews'] = (int)$metricValues[2]->getValue();
                $result['avgSessionDuration'] = (float)$metricValues[3]->getValue();
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log('GA4 Overview Metrics Error: ' . $e->getMessage());
            return [
                'users' => 0,
                'sessions' => 0,
                'pageviews' => 0,
                'avgSessionDuration' => 0
            ];
        }
    }
    
    private function getUAOverviewMetrics($startDate, $endDate) {
        try {
            $response = $this->analytics->data_ga->get(
                'ga:' . $this->viewId,
                $startDate,
                $endDate,
                'ga:users,ga:sessions,ga:pageviews,ga:avgSessionDuration'
            );
            
            $totals = $response->getTotalsForAllResults();
            
            return [
                'users' => (int)$totals['ga:users'],
                'sessions' => (int)$totals['ga:sessions'],
                'pageviews' => (int)$totals['ga:pageviews'],
                'avgSessionDuration' => (float)$totals['ga:avgSessionDuration']
            ];
            
        } catch (Exception $e) {
            error_log('UA Overview Metrics Error: ' . $e->getMessage());
            return [
                'users' => 0,
                'sessions' => 0,
                'pageviews' => 0,
                'avgSessionDuration' => 0
            ];
        }
    }
    
    public function getPageViews($startDate, $endDate, $limit = 10) {
        if ($this->propertyType === 'GA4') {
            return $this->getGA4PageViews($startDate, $endDate, $limit);
        } else {
            return $this->getUAPageViews($startDate, $endDate, $limit);
        }
    }
    
// Enhanced page views with unique page views
    private function getGA4PageViews($startDate, $endDate, $limit) {
    try {
        $request = new Google_Service_AnalyticsData_RunReportRequest();
        $request->setProperty('properties/' . $this->propertyId);
        
        // Set date range
        $dateRange = new Google_Service_AnalyticsData_DateRange();
        $dateRange->setStartDate($startDate);
        $dateRange->setEndDate($endDate);
        $request->setDateRanges([$dateRange]);
        
        // Set dimensions and metrics
        $dimensions = [
            $this->createDimension('pagePath'),
            $this->createDimension('pageTitle')
        ];
        $request->setDimensions($dimensions);
        
        $metrics = [
            $this->createMetric('screenPageViews'),
            $this->createMetric('totalUsers') // Use as proxy for unique page views
        ];
        $request->setMetrics($metrics);
        
        // Set order by
        $orderBy = new Google_Service_AnalyticsData_OrderBy();
        $metricOrderBy = new Google_Service_AnalyticsData_MetricOrderBy();
        $metricOrderBy->setMetricName('screenPageViews');
        $orderBy->setMetric($metricOrderBy);
        $orderBy->setDesc(true);
        $request->setOrderBys([$orderBy]);
        
        $request->setLimit($limit);
        
        $response = $this->analyticsData->properties->runReport('properties/' . $this->propertyId, $request);
        
        $result = [];
        if ($response->getRows()) {
            foreach ($response->getRows() as $row) {
                $dimensionValues = $row->getDimensionValues();
                $metricValues = $row->getMetricValues();
                
                $result[] = [
                    'pagePath' => $dimensionValues[0]->getValue(),
                    'pageTitle' => $dimensionValues[1]->getValue(),
                    'pageviews' => (int)$metricValues[0]->getValue(),
                    'uniquePageviews' => (int)$metricValues[1]->getValue()
                ];
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log('GA4 Page Views Error: ' . $e->getMessage());
        return [];
    }
}
    
    private function getUAPageViews($startDate, $endDate, $limit) {
        try {
            $response = $this->analytics->data_ga->get(
                'ga:' . $this->viewId,
                $startDate,
                $endDate,
                'ga:pageviews',
                [
                    'dimensions' => 'ga:pagePath',
                    'sort' => '-ga:pageviews',
                    'max-results' => $limit
                ]
            );
            
            $result = [];
            foreach ($response->getRows() as $row) {
                $result[] = [
                    'pagePath' => $row[0],
                    'pageviews' => (int)$row[1]
                ];
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log('UA Page Views Error: ' . $e->getMessage());
            return [];
        }
    }
    
    public function getUserCountries($startDate, $endDate, $limit = 10) {
        if ($this->propertyType === 'GA4') {
            return $this->getGA4UserCountries($startDate, $endDate, $limit);
        } else {
            return $this->getUAUserCountries($startDate, $endDate, $limit);
        }
    }
    
    // Enhanced countries with sessions
    private function getGA4UserCountries($startDate, $endDate, $limit) {
    try {
        $request = new Google_Service_AnalyticsData_RunReportRequest();
        $request->setProperty('properties/' . $this->propertyId);
        
        // Set date range
        $dateRange = new Google_Service_AnalyticsData_DateRange();
        $dateRange->setStartDate($startDate);
        $dateRange->setEndDate($endDate);
        $request->setDateRanges([$dateRange]);
        
        // Set dimensions and metrics
        $dimension = $this->createDimension('country');
        $request->setDimensions([$dimension]);
        
        $metrics = [
            $this->createMetric('totalUsers'),
            $this->createMetric('sessions')
        ];
        $request->setMetrics($metrics);
        
        // Set order by
        $orderBy = new Google_Service_AnalyticsData_OrderBy();
        $metricOrderBy = new Google_Service_AnalyticsData_MetricOrderBy();
        $metricOrderBy->setMetricName('totalUsers');
        $orderBy->setMetric($metricOrderBy);
        $orderBy->setDesc(true);
        $request->setOrderBys([$orderBy]);
        
        $request->setLimit($limit);
        
        $response = $this->analyticsData->properties->runReport('properties/' . $this->propertyId, $request);
        
        $result = [];
        if ($response->getRows()) {
            foreach ($response->getRows() as $row) {
                $result[] = [
                    'country' => $row->getDimensionValues()[0]->getValue(),
                    'users' => (int)$row->getMetricValues()[0]->getValue(),
                    'sessions' => (int)$row->getMetricValues()[1]->getValue()
                ];
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log('GA4 User Countries Error: ' . $e->getMessage());
        return [];
    }
}
    
    private function getUAUserCountries($startDate, $endDate, $limit) {
        try {
            $response = $this->analytics->data_ga->get(
                'ga:' . $this->viewId,
                $startDate,
                $endDate,
                'ga:users',
                [
                    'dimensions' => 'ga:country',
                    'sort' => '-ga:users',
                    'max-results' => $limit
                ]
            );
            
            $result = [];
            foreach ($response->getRows() as $row) {
                $result[] = [
                    'country' => $row[0],
                    'users' => (int)$row[1]
                ];
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log('UA User Countries Error: ' . $e->getMessage());
            return [];
        }
    }
    
    public function getDailyTraffic($startDate, $endDate) {
        if ($this->propertyType === 'GA4') {
            return $this->getGA4DailyTraffic($startDate, $endDate);
        } else {
            return $this->getUADailyTraffic($startDate, $endDate);
        }
    }
    
    private function getGA4DailyTraffic($startDate, $endDate) {
        try {
            $request = new Google_Service_AnalyticsData_RunReportRequest();
            $request->setProperty('properties/' . $this->propertyId);
            
            // Set date range
            $dateRange = new Google_Service_AnalyticsData_DateRange();
            $dateRange->setStartDate($startDate);
            $dateRange->setEndDate($endDate);
            $request->setDateRanges([$dateRange]);
            
            // Set dimensions and metrics
            $dimension = new Google_Service_AnalyticsData_Dimension();
            $dimension->setName('date');
            $request->setDimensions([$dimension]);
            
            $metrics = [
                $this->createMetric('totalUsers'),
                $this->createMetric('sessions')
            ];
            $request->setMetrics($metrics);
            
            // Set order by date
            $orderBy = new Google_Service_AnalyticsData_OrderBy();
            $dimensionOrderBy = new Google_Service_AnalyticsData_DimensionOrderBy();
            $dimensionOrderBy->setDimensionName('date');
            $orderBy->setDimension($dimensionOrderBy);
            $request->setOrderBys([$orderBy]);
            
            $response = $this->analyticsData->properties->runReport('properties/' . $this->propertyId, $request);
            
            $result = [];
            foreach ($response->getRows() as $row) {
                $date = $row->getDimensionValues()[0]->getValue();
                // Convert YYYYMMDD to YYYY-MM-DD
                $formattedDate = substr($date, 0, 4) . '-' . substr($date, 4, 2) . '-' . substr($date, 6, 2);
                
                $result[] = [
                    'date' => $formattedDate,
                    'users' => (int)$row->getMetricValues()[0]->getValue(),
                    'sessions' => (int)$row->getMetricValues()[1]->getValue()
                ];
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log('GA4 Daily Traffic Error: ' . $e->getMessage());
            return [];
        }
    }
    
    private function getUADailyTraffic($startDate, $endDate) {
        try {
            $response = $this->analytics->data_ga->get(
                'ga:' . $this->viewId,
                $startDate,
                $endDate,
                'ga:users,ga:sessions',
                [
                    'dimensions' => 'ga:date'
                ]
            );
            
            $result = [];
            foreach ($response->getRows() as $row) {
                $date = $row[0];
                // Convert YYYYMMDD to YYYY-MM-DD
                $formattedDate = substr($date, 0, 4) . '-' . substr($date, 4, 2) . '-' . substr($date, 6, 2);
                
                $result[] = [
                    'date' => $formattedDate,
                    'users' => (int)$row[1],
                    'sessions' => (int)$row[2]
                ];
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log('UA Daily Traffic Error: ' . $e->getMessage());
            return [];
        }
    }
    
    private function createMetric($name) {
        $metric = new Google_Service_AnalyticsData_Metric();
        $metric->setName($name);
        return $metric;
    }
    
    public function getReferrers($startDate, $endDate, $limit = 10) {
    if ($this->propertyType === 'GA4') {
        return $this->getGA4Referrers($startDate, $endDate, $limit);
    } else {
        return $this->getUAReferrers($startDate, $endDate, $limit);
    }
}

    private function getGA4Referrers($startDate, $endDate, $limit) {
    try {
        $request = new Google_Service_AnalyticsData_RunReportRequest();
        $request->setProperty('properties/' . $this->propertyId);
        
        // Set date range
        $dateRange = new Google_Service_AnalyticsData_DateRange();
        $dateRange->setStartDate($startDate);
        $dateRange->setEndDate($endDate);
        $request->setDateRanges([$dateRange]);
        
        // Set dimensions and metrics
        $dimensions = [
            $this->createDimension('sessionSource'),
            $this->createDimension('sessionMedium')
        ];
        $request->setDimensions($dimensions);
        
        $metrics = [
            $this->createMetric('sessions'),
            $this->createMetric('totalUsers')
        ];
        $request->setMetrics($metrics);
        
        // Set order by
        $orderBy = new Google_Service_AnalyticsData_OrderBy();
        $metricOrderBy = new Google_Service_AnalyticsData_MetricOrderBy();
        $metricOrderBy->setMetricName('sessions');
        $orderBy->setMetric($metricOrderBy);
        $orderBy->setDesc(true);
        $request->setOrderBys([$orderBy]);
        
        $request->setLimit($limit);
        
        $response = $this->analyticsData->properties->runReport('properties/' . $this->propertyId, $request);
        
        $result = [];
        if ($response->getRows()) {
            foreach ($response->getRows() as $row) {
                $dimensionValues = $row->getDimensionValues();
                $metricValues = $row->getMetricValues();
                
                $source = $dimensionValues[0]->getValue();
                $medium = $dimensionValues[1]->getValue();
                
                // Combine source and medium for better readability
                $fullSource = $source;
                if ($medium && $medium !== '(none)') {
                    $fullSource .= ' (' . $medium . ')';
                }
                
                $result[] = [
                    'source' => $fullSource,
                    'sessions' => (int)$metricValues[0]->getValue(),
                    'users' => (int)$metricValues[1]->getValue()
                ];
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log('GA4 Referrers Error: ' . $e->getMessage());
        return [];
    }
}

    private function getUAReferrers($startDate, $endDate, $limit) {
    try {
        $response = $this->analytics->data_ga->get(
            'ga:' . $this->viewId,
            $startDate,
            $endDate,
            'ga:sessions,ga:users',
            [
                'dimensions' => 'ga:source,ga:medium',
                'sort' => '-ga:sessions',
                'max-results' => $limit
            ]
        );
        
        $result = [];
        foreach ($response->getRows() as $row) {
            $source = $row[0];
            $medium = $row[1];
            
            // Combine source and medium
            $fullSource = $source;
            if ($medium && $medium !== '(none)') {
                $fullSource .= ' (' . $medium . ')';
            }
            
            $result[] = [
                'source' => $fullSource,
                'sessions' => (int)$row[2],
                'users' => (int)$row[3]
            ];
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log('UA Referrers Error: ' . $e->getMessage());
        return [];
    }
}

    public function getSessionData($startDate, $endDate, $limit = 10) {
    if ($this->propertyType === 'GA4') {
        return $this->getGA4SessionData($startDate, $endDate, $limit);
    } else {
        return $this->getUASessionData($startDate, $endDate, $limit);
    }
}

    private function getGA4SessionData($startDate, $endDate, $limit) {
    try {
        $request = new Google_Service_AnalyticsData_RunReportRequest();
        $request->setProperty('properties/' . $this->propertyId);
        
        // Set date range
        $dateRange = new Google_Service_AnalyticsData_DateRange();
        $dateRange->setStartDate($startDate);
        $dateRange->setEndDate($endDate);
        $request->setDateRanges([$dateRange]);
        
        // Set dimensions and metrics
        $dimension = $this->createDimension('date');
        $request->setDimensions([$dimension]);
        
        $metrics = [
            $this->createMetric('sessions'),
            $this->createMetric('totalUsers'),
            $this->createMetric('bounceRate')
        ];
        $request->setMetrics($metrics);
        
        // Set order by date
        $orderBy = new Google_Service_AnalyticsData_OrderBy();
        $dimensionOrderBy = new Google_Service_AnalyticsData_DimensionOrderBy();
        $dimensionOrderBy->setDimensionName('date');
        $orderBy->setDimension($dimensionOrderBy);
        $orderBy->setDesc(true); // Most recent first
        $request->setOrderBys([$orderBy]);
        
        $request->setLimit($limit);
        
        $response = $this->analyticsData->properties->runReport('properties/' . $this->propertyId, $request);
        
        $result = [];
        if ($response->getRows()) {
            foreach ($response->getRows() as $row) {
                $date = $row->getDimensionValues()[0]->getValue();
                $metricValues = $row->getMetricValues();
                
                // Convert YYYYMMDD to YYYY-MM-DD
                $formattedDate = substr($date, 0, 4) . '-' . substr($date, 4, 2) . '-' . substr($date, 6, 2);
                
                $result[] = [
                    'date' => $formattedDate,
                    'sessions' => (int)$metricValues[0]->getValue(),
                    'users' => (int)$metricValues[1]->getValue(),
                    'bounceRate' => (float)$metricValues[2]->getValue() * 100 // Convert to percentage
                ];
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log('GA4 Session Data Error: ' . $e->getMessage());
        return [];
    }
}

    private function getUASessionData($startDate, $endDate, $limit) {
    try {
        $response = $this->analytics->data_ga->get(
            'ga:' . $this->viewId,
            $startDate,
            $endDate,
            'ga:sessions,ga:users,ga:bounceRate',
            [
                'dimensions' => 'ga:date',
                'sort' => '-ga:date',
                'max-results' => $limit
            ]
        );
        
        $result = [];
        foreach ($response->getRows() as $row) {
            $date = $row[0];
            // Convert YYYYMMDD to YYYY-MM-DD
            $formattedDate = substr($date, 0, 4) . '-' . substr($date, 4, 2) . '-' . substr($date, 6, 2);
            
            $result[] = [
                'date' => $formattedDate,
                'sessions' => (int)$row[1],
                'users' => (int)$row[2],
                'bounceRate' => (float)$row[3]
            ];
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log('UA Session Data Error: ' . $e->getMessage());
        return [];
    }
}

    private function createDimension($name) {
        $dimension = new Google_Service_AnalyticsData_Dimension();
        $dimension->setName($name);
        return $dimension;
    }
    




    
}
?>