<?php
require_once __DIR__.'/config.php';
require_once __DIR__.'/db.php';
require_once __DIR__.'/google_analytics_api.php';

// Verify API request using secret key
verifyApiRequest();

// Get user ID from request
$user_id = isset($_GET['user_id']) ? (int)$_GET['user_id'] : 0;

if (!$user_id) {
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Invalid user ID']);
    exit;
}

// Get Google Analytics credentials from database
$stmt = $pdo->prepare("SELECT * FROM google_analytics_credentials WHERE user_id = ?");
$stmt->execute([$user_id]);
$credentials = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$credentials) {
    header('Content-Type: application/json');
    echo json_encode(['error' => 'No Google Analytics connection found for this user']);
    exit;
}

// Check if token is expired and refresh if needed
if ($credentials['token_expires'] < time()) {
    $new_tokens = refreshAccessToken($credentials['refresh_token']);
    
    if ($new_tokens) {
        // Update tokens in database
        $stmt = $pdo->prepare("
            UPDATE google_analytics_credentials 
            SET access_token = ?, token_expires = ? 
            WHERE user_id = ?
        ");
        $stmt->execute([
            $new_tokens['access_token'],
            time() + $new_tokens['expires_in'],
            $user_id
        ]);
        
        $credentials['access_token'] = $new_tokens['access_token'];
    }
}

// Set date range
$end_date = date('Y-m-d');
$start_date = date('Y-m-d', strtotime('-30 days'));

if (isset($_GET['start_date']) && isset($_GET['end_date'])) {
    $start_date = $_GET['start_date'];
    $end_date = $_GET['end_date'];
}

// Get analytics data
$analytics = new GoogleAnalyticsAPI($credentials['access_token']);

// Determine which data to return based on the 'type' parameter
$type = isset($_GET['type']) ? $_GET['type'] : 'overview';

switch ($type) {
    case 'overview':
        $data = $analytics->getOverviewMetrics(
            $credentials['view_id'],
            $start_date,
            $end_date
        );
        break;
        
    case 'pages':
        $data = $analytics->getPageViews(
            $credentials['view_id'],
            $start_date,
            $end_date
        );
        break;
        
    case 'countries':
        $data = $analytics->getUserCountries(
            $credentials['view_id'],
            $start_date,
            $end_date
        );
        break;
        
    case 'referrers':
        $data = $analytics->getReferrers(
            $credentials['view_id'],
            $start_date,
            $end_date
        );
        break;
        
    case 'daily':
        $data = $analytics->getDailyTraffic(
            $credentials['view_id'],
            $start_date,
            $end_date
        );
        break;
        
    default:
        $data = [
            'overview' => $analytics->getOverviewMetrics($credentials['view_id'], $start_date, $end_date),
            'pages' => $analytics->getPageViews($credentials['view_id'], $start_date, $end_date, 5),
            'countries' => $analytics->getUserCountries($credentials['view_id'], $start_date, $end_date, 5),
            'referrers' => $analytics->getReferrers($credentials['view_id'], $start_date, $end_date, 5)
        ];
}

// Return data as JSON
header('Content-Type: application/json');
echo json_encode($data);
?>