<script>

// Set PDF.js worker
pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';

class EnhancedPDFViewer {
  constructor() {
    // PHP integrated PDF URL
    this.defaultPdfUrl = "<?php echo APP_URL.$pdfPath;?>";
    
    this.currentPage = 1;
    this.totalPages = 1;
    this.scale = 0.85;
    this.minScale = 0.5;
    this.maxScale = 4.0;
    this.rotation = 0;
    this.soundEnabled = true;
    this.isDarkTheme = false;
    this.isFullscreen = false;
    this.isDualPageMode = false; // Changed to false - always single page
    this.isSidePanelOpen = false;
    this.pdfDoc = null;
    this.pageRendering = false;
    this.pageNumPending = null;
    this.isMobile = window.innerWidth <= 768;

    // Enhanced zoom and pan variables
    this.isZooming = false;
    this.isPanning = false;
    this.panX = 0;
    this.panY = 0;
    this.lastTouchDistance = 0;
    this.touchStartX = 0;
    this.touchStartY = 0;
    this.initialPinchDistance = 0;
    this.initialScale = 1.0;
    this.touches = [];
    
    // Enhanced pan variables
    this.isDragging = false;
    this.dragStartX = 0;
    this.dragStartY = 0;
    this.lastPanX = 0;
    this.lastPanY = 0;
    this.panVelocityX = 0;
    this.panVelocityY = 0;
    this.panInertia = false;

    this.initializeElements();
    this.bindEvents();
    this.setupKeyboardShortcuts();
    this.setupAdvancedTouchGestures();
    this.setupAdvancedMouseControls();
    this.showWelcomeMessage();

    // Auto-load default PDF if provided
    if (this.defaultPdfUrl && this.defaultPdfUrl.trim() !== '') {
      this.loadPDF(this.defaultPdfUrl);
    } else {
      this.hideLoading();
      this.updateStatus('No PDF specified. Click "Load PDF" to select a file.');
    }

    // Handle window resize
    window.addEventListener('resize', () => {
      this.isMobile = window.innerWidth <= 768;
      this.updatePageLayout();
    });

    // Create zoom indicator
    this.createZoomIndicator();
  }

  createZoomIndicator() {
    this.zoomIndicator = document.createElement('div');
    this.zoomIndicator.className = 'shu-pdf-zoom-indicator';
    this.zoomIndicator.textContent = '100%';
    this.wrapper?.appendChild(this.zoomIndicator);
  }

  initializeElements() {
    // Control buttons
    this.firstPageBtn = document.getElementById('shu-pdf-firstPageBtn');
    this.prevPageBtn = document.getElementById('shu-pdf-prevPageBtn');
    this.nextPageBtn = document.getElementById('shu-pdf-nextPageBtn');
    this.lastPageBtn = document.getElementById('shu-pdf-lastPageBtn');
    this.zoomInBtn = document.getElementById('shu-pdf-zoomInBtn');
    this.zoomOutBtn = document.getElementById('shu-pdf-zoomOutBtn');
    this.zoomAllReset = document.getElementById('shu-pdf-resetZoomBtn');
    this.fitWidthBtn = document.getElementById('shu-pdf-fitWidthBtn');
    this.fitPageBtn = document.getElementById('shu-pdf-fitPageBtn');
    this.dualPageBtn = document.getElementById('shu-pdf-dualPageBtn');
    this.fullscreenBtn = document.getElementById('shu-pdf-fullscreenBtn');
    this.rotateBtn = document.getElementById('shu-pdf-rotateBtn');
    this.loadFileBtn = document.getElementById('shu-pdf-loadFileBtn');
    this.soundToggleBtn = document.getElementById('shu-pdf-soundToggleBtn');
    this.themeToggleBtn = document.getElementById('shu-pdf-themeToggleBtn');

    // Side panel elements
    this.toggleSidePanelBtn = document.getElementById('shu-pdf-toggleSidePanel');
    this.closeSidePanelBtn = document.getElementById('shu-pdf-closeSidePanel');
    this.sidePanel = document.getElementById('shu-pdf-sidePanel');
    this.pageThumbnails = document.getElementById('shu-pdf-pageThumbnails');

    // Navigation arrows
    this.navPrevBtn = document.getElementById('shu-pdf-navPrevBtn');
    this.navNextBtn = document.getElementById('shu-pdf-navNextBtn');

    // Canvas and container elements
    this.leftCanvas = document.getElementById('shu-pdf-leftCanvas');
    this.rightCanvas = document.getElementById('shu-pdf-rightCanvas');
    this.leftCtx = this.leftCanvas?.getContext('2d');
    this.rightCtx = this.rightCanvas?.getContext('2d');
    this.pagesContainer = document.getElementById('shu-pdf-pagesContainer');
    this.leftPageContainer = document.getElementById('shu-pdf-leftPageContainer');
    this.rightPageContainer = document.getElementById('shu-pdf-rightPageContainer');
    this.wrapper = document.getElementById('shu-pdf-wrapper');

    // Other elements
    this.currentPageInput = document.getElementById('shu-pdf-currentPageInput');
    this.totalPagesSpan = document.getElementById('shu-pdf-totalPages');
    this.zoomLevelSpan = document.getElementById('shu-pdf-zoomLevel');
    this.statusText = document.getElementById('shu-pdf-statusText');
    this.loadingOverlay = document.getElementById('shu-pdf-loadingOverlay');
    this.pdfContainer = document.getElementById('shu-pdf-pdfContainer');
    this.themeText = document.getElementById('shu-pdf-themeText');
    this.fileInput = document.getElementById('shu-pdf-fileInput');
    this.pageTurnSound = document.getElementById('shu-pdf-pageTurnSound');
    this.viewerContainer = document.querySelector('.shu-pdf-viewer-container');

    // Hide dual page button since we're always in single page mode
    if (this.dualPageBtn) {
      this.dualPageBtn.style.display = 'none';
    }
  }

  bindEvents() {
    // Navigation buttons
    this.firstPageBtn?.addEventListener('click', () => this.goToFirstPage());
    this.prevPageBtn?.addEventListener('click', () => this.previousPage());
    this.nextPageBtn?.addEventListener('click', () => this.nextPage());
    this.lastPageBtn?.addEventListener('click', () => this.goToLastPage());

    // Navigation arrows
    this.navPrevBtn?.addEventListener('click', () => this.previousPage());
    this.navNextBtn?.addEventListener('click', () => this.nextPage());

    // Zoom controls
    this.zoomInBtn?.addEventListener('click', () => this.zoomIn());
    this.zoomOutBtn?.addEventListener('click', () => this.zoomOut());
    this.zoomAllReset?.addEventListener('click', () => this.resetZoom());
    this.fitWidthBtn?.addEventListener('click', () => this.fitWidth());
    this.fitPageBtn?.addEventListener('click', () => this.fitPage());

    // Side panel controls
    this.toggleSidePanelBtn?.addEventListener('click', () => this.toggleSidePanel());
    this.closeSidePanelBtn?.addEventListener('click', () => this.closeSidePanel());

    // Other controls
    this.fullscreenBtn?.addEventListener('click', () => this.toggleFullscreen());
    this.rotateBtn?.addEventListener('click', () => this.rotate());
    this.loadFileBtn?.addEventListener('click', () => this.fileInput?.click());
    this.soundToggleBtn?.addEventListener('click', () => this.toggleSound());
    this.themeToggleBtn?.addEventListener('click', () => this.toggleTheme());

    // File input
    this.fileInput?.addEventListener('change', (e) => this.handleFileLoad(e));

    // Page input
    this.currentPageInput?.addEventListener('change', () => this.goToPage());
    this.currentPageInput?.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') this.goToPage();
    });

    // Mouse wheel zoom
    this.wrapper?.addEventListener('wheel', (e) => this.handleWheelZoom(e));

    // Fullscreen change events
    document.addEventListener('fullscreenchange', () => this.onFullscreenChange());
    document.addEventListener('webkitfullscreenchange', () => this.onFullscreenChange());
    document.addEventListener('mozfullscreenchange', () => this.onFullscreenChange());
    document.addEventListener('MSFullscreenChange', () => this.onFullscreenChange());
  }

  setupKeyboardShortcuts() {
    document.addEventListener('keydown', (e) => {
      if (e.target.tagName !== 'INPUT') {
        // Enhanced keyboard navigation for zoomed content
        if (this.scale > 1.0) {
          switch (e.key) {
            case 'ArrowUp':
              e.preventDefault();
              this.panUp();
              return;
            case 'ArrowDown':
              e.preventDefault();
              this.panDown();
              return;
            case 'ArrowLeft':
              e.preventDefault();
              this.panLeft();
              return;
            case 'ArrowRight':
              e.preventDefault();
              this.panRight();
              return;
          }
        }

        // Standard navigation when not zoomed
        switch (e.key) {
          case 'ArrowLeft':
          case 'PageUp':
            e.preventDefault();
            this.previousPage();
            break;
          case 'ArrowRight':
          case 'PageDown':
          case ' ':
            e.preventDefault();
            this.nextPage();
            break;
          case 'Home':
            e.preventDefault();
            this.goToFirstPage();
            break;
          case 'End':
            e.preventDefault();
            this.goToLastPage();
            break;
          case '+':
          case '=':
            e.preventDefault();
            this.zoomIn();
            break;
          case '-':
            e.preventDefault();
            this.zoomOut();
            break;
          case '0':
            e.preventDefault();
            this.resetZoom();
            break;
          case 'f':
          case 'F11':
            e.preventDefault();
            this.toggleFullscreen();
            break;
          case 's':
            e.preventDefault();
            this.toggleSidePanel();
            break;
          case 'r':
            e.preventDefault();
            this.rotate();
            break;
          case 't':
            e.preventDefault();
            this.toggleTheme();
            break;
          case 'Escape':
            if (this.isFullscreen) {
              e.preventDefault();
              this.exitFullscreen();
            } else if (this.isSidePanelOpen) {
              e.preventDefault();
              this.closeSidePanel();
            } else if (this.scale > 1.0) {
              e.preventDefault();
              this.resetZoom();
            }
            break;
        }
      }
    });
  }

  // Enhanced pan methods for keyboard navigation
  panUp() {
    this.panY += 50;
    this.applyTransform();
    this.showPanFeedback('⬆');
  }

  panDown() {
    this.panY -= 50;
    this.applyTransform();
    this.showPanFeedback('⬇');
  }

  panLeft() {
    this.panX += 50;
    this.applyTransform();
    this.showPanFeedback('⬅');
  }

  panRight() {
    this.panX -= 50;
    this.applyTransform();
    this.showPanFeedback('➡');
  }

  showPanFeedback(direction) {
    if (this.zoomIndicator) {
      this.zoomIndicator.textContent = `${Math.round(this.scale * 100)}% ${direction}`;
      this.zoomIndicator.classList.add('visible');
      
      clearTimeout(this.panFeedbackTimeout);
      this.panFeedbackTimeout = setTimeout(() => {
        this.zoomIndicator.textContent = `${Math.round(this.scale * 100)}%`;
        this.zoomIndicator.classList.remove('visible');
      }, 800);
    }
  }

  setupAdvancedTouchGestures() {
    if (!this.wrapper) return;

    // Touch start
    this.wrapper.addEventListener('touchstart', (e) => {
      this.touches = Array.from(e.touches);
      
      if (this.touches.length === 1) {
        // Single touch - prepare for pan
        const touch = this.touches[0];
        this.touchStartX = touch.clientX;
        this.touchStartY = touch.clientY;
        this.isPanning = false;
        this.lastPanX = this.panX;
        this.lastPanY = this.panY;
      } else if (this.touches.length === 2) {
        // Two finger touch - prepare for zoom
        e.preventDefault();
        this.isZooming = true;
        this.initialPinchDistance = this.getTouchDistance(this.touches[0], this.touches[1]);
        this.initialScale = this.scale;
        
        // Get center point for zoom
        const centerX = (this.touches[0].clientX + this.touches[1].clientX) / 2;
        const centerY = (this.touches[0].clientY + this.touches[1].clientY) / 2;
        this.zoomCenterX = centerX;
        this.zoomCenterY = centerY;
      }
    });

    // Touch move
    this.wrapper.addEventListener('touchmove', (e) => {
      if (this.touches.length === 1 && !this.isZooming) {
        // Single finger pan
        const touch = e.touches[0];
        const deltaX = touch.clientX - this.touchStartX;
        const deltaY = touch.clientY - this.touchStartY;
        
        if (Math.abs(deltaX) > 5 || Math.abs(deltaY) > 5) {
          this.isPanning = true;
          
          if (this.scale > 1.0) {
            // Enhanced pan when zoomed in
            e.preventDefault();
            this.panX = this.lastPanX + deltaX;
            this.panY = this.lastPanY + deltaY;
            this.applyTransform();
          }
        }
      } else if (this.touches.length === 2 && this.isZooming) {
        // Two finger zoom
        e.preventDefault();
        const currentDistance = this.getTouchDistance(e.touches[0], e.touches[1]);
        const scaleChange = currentDistance / this.initialPinchDistance;
        const newScale = Math.max(this.minScale, Math.min(this.maxScale, this.initialScale * scaleChange));
        
        if (Math.abs(newScale - this.scale) > 0.02) {
          this.scale = newScale;
          this.applyTransform();
          this.updateZoomDisplay();
          this.showZoomIndicator();
        }
      }
    });

    // Touch end
    this.wrapper.addEventListener('touchend', (e) => {
      if (this.isZooming) {
        this.isZooming = false;
        this.renderCurrentPages(); // Re-render at new scale
        return;
      }

      if (!this.isPanning && this.touches.length === 1 && this.scale <= 1.0) {
        // Single tap without pan and not zoomed - check for page navigation
        const touch = e.changedTouches[0];
        const rect = this.wrapper.getBoundingClientRect();
        const x = touch.clientX - rect.left;
        const width = rect.width;
        
        if (x < width * 0.3) {
          this.previousPage();
        } else if (x > width * 0.7) {
          this.nextPage();
        }
      }
      
      this.isPanning = false;
      this.touches = [];
    });
  }

  setupAdvancedMouseControls() {
    if (!this.wrapper) return;

    let isMouseDown = false;
    let mouseStartX = 0;
    let mouseStartY = 0;
    let lastMouseX = 0;
    let lastMouseY = 0;

    // Mouse down
    this.wrapper.addEventListener('mousedown', (e) => {
      if (this.scale > 1.0) {
        isMouseDown = true;
        this.isDragging = true;
        mouseStartX = e.clientX;
        mouseStartY = e.clientY;
        lastMouseX = e.clientX;
        lastMouseY = e.clientY;
        this.dragStartX = this.panX;
        this.dragStartY = this.panY;
        this.wrapper.style.cursor = 'grabbing';
        e.preventDefault();
      }
    });

    // Mouse move
    this.wrapper.addEventListener('mousemove', (e) => {
      if (isMouseDown && this.scale > 1.0) {
        const deltaX = e.clientX - mouseStartX;
        const deltaY = e.clientY - mouseStartY;
        
        // Calculate velocity for potential inertia
        this.panVelocityX = e.clientX - lastMouseX;
        this.panVelocityY = e.clientY - lastMouseY;
        
        this.panX = this.dragStartX + deltaX;
        this.panY = this.dragStartY + deltaY;
        this.applyTransform();
        
        lastMouseX = e.clientX;
        lastMouseY = e.clientY;
      } else if (this.scale > 1.0) {
        // Change cursor when hovering over zoomable content
        this.wrapper.style.cursor = 'grab';
      } else {
        this.wrapper.style.cursor = 'default';
      }
    });

    // Mouse up
    this.wrapper.addEventListener('mouseup', (e) => {
      if (isMouseDown) {
        isMouseDown = false;
        this.isDragging = false;
        
        // Apply inertia if there was significant velocity
        if (Math.abs(this.panVelocityX) > 5 || Math.abs(this.panVelocityY) > 5) {
          this.applyPanInertia();
        }
        
        this.wrapper.style.cursor = this.scale > 1.0 ? 'grab' : 'default';
      }
    });

    // Mouse leave
    this.wrapper.addEventListener('mouseleave', () => {
      if (isMouseDown) {
        isMouseDown = false;
        this.isDragging = false;
        this.wrapper.style.cursor = 'default';
      }
    });

    // Double click to zoom
    this.wrapper.addEventListener('dblclick', (e) => {
      e.preventDefault();
      if (this.scale === 1.0) {
        this.scale = 2.0;
        
        // Zoom to the clicked point
        const rect = this.wrapper.getBoundingClientRect();
        const clickX = e.clientX - rect.left - rect.width / 2;
        const clickY = e.clientY - rect.top - rect.height / 2;
        
        this.panX = -clickX * 0.5;
        this.panY = -clickY * 0.5;
      } else {
        this.resetZoom();
      }
      
      this.applyTransform();
      this.updateZoomDisplay();
      this.showZoomIndicator();
      this.renderCurrentPages();
    });
  }

  applyPanInertia() {
    if (!this.panInertia) {
      this.panInertia = true;
      const friction = 0.95;
      const minVelocity = 0.5;
      
      const animate = () => {
        this.panVelocityX *= friction;
        this.panVelocityY *= friction;
        
        this.panX += this.panVelocityX;
        this.panY += this.panVelocityY;
        
        this.applyTransform();
        
        if (Math.abs(this.panVelocityX) > minVelocity || Math.abs(this.panVelocityY) > minVelocity) {
          requestAnimationFrame(animate);
        } else {
          this.panInertia = false;
        }
      };
      
      requestAnimationFrame(animate);
    }
  }

  handleWheelZoom(e) {
    e.preventDefault();
    
    if (e.ctrlKey || e.metaKey) {
      // Zoom with Ctrl/Cmd + wheel
      const delta = e.deltaY > 0 ? -0.1 : 0.1;
      const newScale = Math.max(this.minScale, Math.min(this.maxScale, this.scale + delta));
      
      if (newScale !== this.scale) {
        // Zoom towards mouse position
        const rect = this.wrapper.getBoundingClientRect();
        const mouseX = e.clientX - rect.left - rect.width / 2;
        const mouseY = e.clientY - rect.top - rect.height / 2;
        
        const scaleDiff = newScale / this.scale;
        this.panX = this.panX * scaleDiff - mouseX * (scaleDiff - 1) * 0.1;
        this.panY = this.panY * scaleDiff - mouseY * (scaleDiff - 1) * 0.1;
        
        this.scale = newScale;
        this.applyTransform();
        this.updateZoomDisplay();
        this.showZoomIndicator();
        
        // Debounce re-rendering
        clearTimeout(this.renderTimeout);
        this.renderTimeout = setTimeout(() => {
          this.renderCurrentPages();
        }, 300);
      }
    } else if (this.scale > 1.0) {
      // Pan with wheel when zoomed in
      const panSpeed = 30;
      if (e.shiftKey) {
        // Horizontal pan with Shift + wheel
        this.panX += e.deltaY > 0 ? -panSpeed : panSpeed;
      } else {
        // Vertical pan with wheel
        this.panY += e.deltaY > 0 ? -panSpeed : panSpeed;
      }
      this.applyTransform();
    }
  }

  getTouchDistance(touch1, touch2) {
    const dx = touch1.clientX - touch2.clientX;
    const dy = touch1.clientY - touch2.clientY;
    return Math.sqrt(dx * dx + dy * dy);
  }

  applyTransform() {
    if (this.pagesContainer) {
      // Calculate dynamic pan bounds based on scale and container size
      const containerRect = this.wrapper?.getBoundingClientRect();
      const containerWidth = containerRect?.width || 1200;
      const containerHeight = containerRect?.height || 800;
      
      // More generous pan bounds when zoomed in
      const maxPanX = Math.max(200, containerWidth * this.scale * 0.3);
      const maxPanY = Math.max(200, containerHeight * this.scale * 0.3);
      
      // Constrain pan to calculated bounds
      this.panX = Math.max(-maxPanX, Math.min(maxPanX, this.panX));
      this.panY = Math.max(-maxPanY, Math.min(maxPanY, this.panY));
      
      this.pagesContainer.style.transform = `scale(${this.scale}) translate(${this.panX}px, ${this.panY}px)`;
      this.pagesContainer.style.transformOrigin = 'center center';
      
      // Update wrapper class for cursor
      if (this.scale > 1.0) {
        this.wrapper.classList.add('zoomed');
      } else {
        this.wrapper.classList.remove('zoomed');
        this.panX = 0;
        this.panY = 0;
      }
    }
  }

  showZoomIndicator() {
    if (this.zoomIndicator) {
      this.zoomIndicator.textContent = `${Math.round(this.scale * 100)}%`;
      this.zoomIndicator.classList.add('visible');
      
      clearTimeout(this.zoomIndicatorTimeout);
      this.zoomIndicatorTimeout = setTimeout(() => {
        this.zoomIndicator.classList.remove('visible');
      }, 1500);
    }
  }

  updateZoomDisplay() {
    if (this.zoomLevelSpan) {
      this.zoomLevelSpan.textContent = `${Math.round(this.scale * 100)}%`;
    }
  }

  async renderCurrentPages() {
    if (!this.pdfDoc) return;

    this.updatePageLayout();
    
    // Always render in single page mode
    await this.renderPage(this.currentPage, this.leftCanvas, this.leftCtx);
    
    // Hide the right page container since we're in single page mode
    if (this.rightPageContainer) {
      this.rightPageContainer.style.display = 'none';
    }

    // Apply current transform
    this.applyTransform();
  }

  async renderPage(pageNum, canvas, ctx) {
    if (!this.pdfDoc || !canvas || !ctx || pageNum < 1 || pageNum > this.totalPages) {
      return;
    }

    try {
      const page = await this.pdfDoc.getPage(pageNum);
      
      // Calculate optimal scale for single page display
      let baseScale = 1.5; // Base scale for quality
      
      // Calculate scale to fit the single page optimally
      const containerWidth = this.wrapper?.clientWidth || 1200;
      const containerHeight = this.wrapper?.clientHeight || 800;
      
      // Use more of the available space for single page
      const availableWidth = containerWidth - 60; // Account for padding
      const availableHeight = containerHeight - 60; // Account for padding
      
      const viewport = page.getViewport({ scale: 1.0 });
      const scaleX = availableWidth / viewport.width;
      const scaleY = availableHeight / viewport.height;
      
      baseScale = Math.min(scaleX, scaleY, 2.5); // Increased max scale for better quality
      
      const finalViewport = page.getViewport({
        scale: baseScale,
        rotation: this.rotation
      });

      canvas.height = finalViewport.height;
      canvas.width = finalViewport.width;

      const renderContext = {
        canvasContext: ctx,
        viewport: finalViewport
      };

      await page.render(renderContext).promise;
      
      // Show the canvas
      canvas.style.display = 'block';
    } catch (error) {
      console.error('Error rendering page:', error);
    }
  }

  clearCanvas(canvas, ctx) {
    if (canvas && ctx) {
      ctx.clearRect(0, 0, canvas.width, canvas.height);
      canvas.style.display = 'none';
    }
  }

  updatePageLayout() {
    if (!this.pagesContainer) return;
    
    // Always use single column layout
    this.pagesContainer.style.flexDirection = 'column';
    this.pagesContainer.style.alignItems = 'center';
    
    if (this.rightPageContainer) {
      this.rightPageContainer.style.display = 'none';
    }
  }

  async zoomIn() {
    if (this.scale < this.maxScale) {
      this.scale = Math.min(this.maxScale, this.scale + 0.15);
      this.applyTransform();
      this.updateZoomDisplay();
      this.showZoomIndicator();
      await this.renderCurrentPages();
    }
  }

  async zoomOut() {
    if (this.scale > this.minScale) {
      this.scale = Math.max(this.minScale, this.scale - 0.15);
      this.applyTransform();
      this.updateZoomDisplay();
      this.showZoomIndicator();
      await this.renderCurrentPages();
    }
  }

  async resetZoom() {
    this.scale = 1.0;
    this.panX = 0;
    this.panY = 0;
    this.applyTransform();
    this.updateZoomDisplay();
    this.showZoomIndicator();
    await this.renderCurrentPages();
    this.updateStatus('Zoom reset to 100%');
  }

  showWelcomeMessage() {
    if (this.defaultPdfUrl && this.defaultPdfUrl.trim() !== '') {
      this.updateStatus('Loading PDF...');
    } else {
      this.updateStatus('Welcome! Click "Load PDF" to start viewing your document.');
    }
  }

  handleFileLoad(event) {
    const file = event.target.files[0];
    if (file && file.type === 'application/pdf') {
      this.showLoading();
      this.updateStatus('Loading PDF file...');
      const fileURL = URL.createObjectURL(file);
      this.loadPDF(fileURL);
    } else {
      this.updateStatus('Please select a valid PDF file.');
    }
  }

  async loadPDF(url) {
    try {
      this.showLoading();
      this.updateStatus('Loading PDF...');
      
      let loadingTask;
      
      try {
        // First attempt - standard loading
        loadingTask = pdfjsLib.getDocument({
          url: url,
          cMapUrl: 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/cmaps/',
          cMapPacked: true,
          withCredentials: false,
          disableAutoFetch: false,
          disableStream: false,
          disableRange: false
        });
      } catch (firstError) {
        console.log('First loading attempt failed, trying alternative method:', firstError);
        
        // Second attempt - with different settings
        loadingTask = pdfjsLib.getDocument({
          url: url,
          cMapUrl: 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/cmaps/',
          cMapPacked: true,
          withCredentials: true,
          disableAutoFetch: true,
          disableStream: true,
          disableRange: true
        });
      }

      loadingTask.onProgress = (progress) => {
        if (progress.total > 0) {
          const percent = Math.round((progress.loaded / progress.total) * 100);
          this.updateStatus(`Loading PDF... ${percent}%`);
        }
      };

      this.pdfDoc = await loadingTask.promise;
      this.totalPages = this.pdfDoc.numPages;
      
      if (this.totalPagesSpan) {
        this.totalPagesSpan.textContent = this.totalPages;
      }

      this.currentPage = 1;
      if (this.currentPageInput) {
        this.currentPageInput.value = 1;
      }

      await this.renderCurrentPages();
      await this.generateThumbnails();
      this.hideLoading();
      this.updateStatus(`PDF loaded successfully. ${this.totalPages} pages. Single page view mode.`);
      this.updateButtonStates();
      
    } catch (error) {
      this.hideLoading();
      const errorMessage = `Error loading PDF: ${error.message}`;
      this.updateStatus(errorMessage);
      console.error('PDF loading error:', error);
      
      // Provide more specific error information
      if (error.name === 'InvalidPDFException') {
        this.updateStatus('Error: Invalid PDF file format.');
      } else if (error.name === 'MissingPDFException') {
        this.updateStatus('Error: PDF file not found. Please check the file path.');
      } else if (error.name === 'UnexpectedResponseException') {
        this.updateStatus('Error: Unable to load PDF. Please check your internet connection or file permissions.');
      } else if (error.message.includes('CORS')) {
        this.updateStatus('Error: Cross-origin request blocked. Please check server configuration.');
      }
    }
  }

  async generateThumbnails() {
    if (!this.pdfDoc || !this.pageThumbnails) return;

    this.pageThumbnails.innerHTML = '';

    for (let i = 1; i <= this.totalPages; i++) {
      const thumbnailDiv = document.createElement('div');
      thumbnailDiv.className = 'shu-pdf-thumbnail';
      if (i === this.currentPage) {
        thumbnailDiv.classList.add('active');
      }

      const canvas = document.createElement('canvas');
      canvas.className = 'shu-pdf-thumbnail-canvas';
      
      const ctx = canvas.getContext('2d');
      
      try {
        const page = await this.pdfDoc.getPage(i);
        const viewport = page.getViewport({ scale: 0.2 });
        
        canvas.height = viewport.height;
        canvas.width = viewport.width;

        await page.render({
          canvasContext: ctx,
          viewport: viewport
        }).promise;
      } catch (error) {
        console.error('Error generating thumbnail for page', i, error);
      }

      const pageInfo = document.createElement('div');
      pageInfo.innerHTML = `<strong>Page ${i}</strong>`;

      thumbnailDiv.appendChild(canvas);
      thumbnailDiv.appendChild(pageInfo);

      thumbnailDiv.addEventListener('click', () => {
        this.goToSpecificPage(i);
        this.updateThumbnailSelection(i);
      });

      this.pageThumbnails.appendChild(thumbnailDiv);
    }
  }

  updateThumbnailSelection(pageNum) {
    const thumbnails = this.pageThumbnails?.querySelectorAll('.shu-pdf-thumbnail');
    thumbnails?.forEach((thumb, index) => {
      if (index + 1 === pageNum) {
        thumb.classList.add('active');
      } else {
        thumb.classList.remove('active');
      }
    });
  }

  toggleSidePanel() {
    this.isSidePanelOpen = !this.isSidePanelOpen;
    if (this.sidePanel) {
      this.sidePanel.classList.toggle('open', this.isSidePanelOpen);
    }
  }

  closeSidePanel() {
    this.isSidePanelOpen = false;
    if (this.sidePanel) {
      this.sidePanel.classList.remove('open');
    }
  }

  showLoading() {
    if (this.loadingOverlay) {
      this.loadingOverlay.classList.remove('hidden');
    }
  }

  hideLoading() {
    if (this.loadingOverlay) {
      this.loadingOverlay.classList.add('hidden');
    }
  }

  updateStatus(message) {
    if (this.statusText) {
      this.statusText.textContent = message;
    }
  }

  playPageTurnSound() {
    if (this.soundEnabled && this.pageTurnSound) {
      this.pageTurnSound.currentTime = 0;
      this.pageTurnSound.play().catch(() => {
        // Ignore audio play errors
      });
    }
  }

  addPageTurnAnimation() {
    if (this.pdfContainer) {
      this.pdfContainer.classList.add('shu-pdf-page-turning');
      setTimeout(() => {
        this.pdfContainer.classList.remove('shu-pdf-page-turning');
      }, 600);
    }
  }

  async nextPage() {
    if (this.currentPage < this.totalPages) {
      this.currentPage++;
      if (this.currentPageInput) {
        this.currentPageInput.value = this.currentPage;
      }
      await this.renderCurrentPages();
      this.updateButtonStates();
      this.updateThumbnailSelection(this.currentPage);
      this.playPageTurnSound();
      this.addPageTurnAnimation();
      this.updateStatus(`Page ${this.currentPage} of ${this.totalPages}`);
    }
  }

  async previousPage() {
    if (this.currentPage > 1) {
      this.currentPage--;
      if (this.currentPageInput) {
        this.currentPageInput.value = this.currentPage;
      }
      await this.renderCurrentPages();
      this.updateButtonStates();
      this.updateThumbnailSelection(this.currentPage);
      this.playPageTurnSound();
      this.addPageTurnAnimation();
      this.updateStatus(`Page ${this.currentPage} of ${this.totalPages}`);
    }
  }

  async goToFirstPage() {
    if (this.currentPage !== 1) {
      this.currentPage = 1;
      if (this.currentPageInput) {
        this.currentPageInput.value = this.currentPage;
      }
      await this.renderCurrentPages();
      this.updateButtonStates();
      this.updateThumbnailSelection(this.currentPage);
      this.playPageTurnSound();
      this.addPageTurnAnimation();
      this.updateStatus(`Page ${this.currentPage} of ${this.totalPages}`);
    }
  }

  async goToLastPage() {
    if (this.currentPage !== this.totalPages) {
      this.currentPage = this.totalPages;
      if (this.currentPageInput) {
        this.currentPageInput.value = this.currentPage;
      }
      await this.renderCurrentPages();
      this.updateButtonStates();
      this.updateThumbnailSelection(this.currentPage);
      this.playPageTurnSound();
      this.addPageTurnAnimation();
      this.updateStatus(`Page ${this.currentPage} of ${this.totalPages}`);
    }
  }

  async goToPage() {
    if (!this.currentPageInput) return;
    
    const pageNum = parseInt(this.currentPageInput.value);
    if (pageNum >= 1 && pageNum <= this.totalPages && pageNum !== this.currentPage) {
      this.currentPage = pageNum;
      await this.renderCurrentPages();
      this.updateButtonStates();
      this.updateThumbnailSelection(this.currentPage);
      this.playPageTurnSound();
      this.addPageTurnAnimation();
      this.updateStatus(`Page ${this.currentPage} of ${this.totalPages}`);
    } else {
      this.currentPageInput.value = this.currentPage;
    }
  }

  async goToSpecificPage(pageNum) {
    if (pageNum >= 1 && pageNum <= this.totalPages && pageNum !== this.currentPage) {
      this.currentPage = pageNum;
      if (this.currentPageInput) {
        this.currentPageInput.value = this.currentPage;
      }
      await this.renderCurrentPages();
      this.updateButtonStates();
      this.playPageTurnSound();
      this.addPageTurnAnimation();
      this.updateStatus(`Page ${this.currentPage} of ${this.totalPages}`);
    }
  }

  updateButtonStates() {
    if (this.prevPageBtn) this.prevPageBtn.disabled = this.currentPage === 1;
    if (this.firstPageBtn) this.firstPageBtn.disabled = this.currentPage === 1;
    if (this.nextPageBtn) this.nextPageBtn.disabled = this.currentPage === this.totalPages;
    if (this.lastPageBtn) this.lastPageBtn.disabled = this.currentPage === this.totalPages;
    
    if (this.navPrevBtn) {
      this.navPrevBtn.style.opacity = this.currentPage === 1 ? '0.5' : '1';
    }
    if (this.navNextBtn) {
      this.navNextBtn.style.opacity = this.currentPage === this.totalPages ? '0.5' : '1';
    }
  }

  async fitWidth() {
    this.scale = 1.0;
    await this.updateZoom();
    this.updateStatus('Fit to width');
  }

  async fitPage() {
    this.scale = 0.85;
    await this.updateZoom();
    this.updateStatus('Fit to page');
  }

  async updateZoom() {
    if (this.zoomLevelSpan) {
      this.zoomLevelSpan.textContent = `${Math.round(this.scale * 100)}%`;
    }
    if (this.pdfDoc) {
      await this.renderCurrentPages();
    }
    this.updateStatus(`Zoom: ${Math.round(this.scale * 100)}%`);
  }

  async rotate() {
    this.rotation = (this.rotation + 90) % 360;
    if (this.pdfDoc) {
      await this.renderCurrentPages();
    }
    this.updateStatus(`Rotated ${this.rotation}°`);
  }

  toggleFullscreen() {
    if (!this.isFullscreen) {
      this.enterFullscreen();
    } else {
      this.exitFullscreen();
    }
  }

  enterFullscreen() {
    const element = this.pdfContainer;
    if (!element) return;

    if (element.requestFullscreen) {
      element.requestFullscreen();
    } else if (element.webkitRequestFullscreen) {
      element.webkitRequestFullscreen();
    } else if (element.mozRequestFullScreen) {
      element.mozRequestFullScreen();
    } else if (element.msRequestFullscreen) {
      element.msRequestFullscreen();
    }
  }

  exitFullscreen() {
    if (document.exitFullscreen) {
      document.exitFullscreen();
    } else if (document.webkitExitFullscreen) {
      document.webkitExitFullscreen();
    } else if (document.mozCancelFullScreen) {
      document.mozCancelFullScreen();
    } else if (document.msExitFullscreen) {
      document.msExitFullscreen();
    }
  }

  onFullscreenChange() {
    this.isFullscreen = !!(
      document.fullscreenElement ||
      document.webkitFullscreenElement ||
      document.mozFullScreenElement ||
      document.msFullscreenElement
    );

    if (this.isFullscreen) {
      if (this.pdfContainer) {
        this.pdfContainer.classList.add('shu-pdf-fullscreen');
      }
      if (this.fullscreenBtn) {
        this.fullscreenBtn.innerHTML = '<i class="fas fa-compress"></i>';
      }
      this.updateStatus('Entered fullscreen mode');
    } else {
      if (this.pdfContainer) {
        this.pdfContainer.classList.remove('shu-pdf-fullscreen');
      }
      if (this.fullscreenBtn) {
        this.fullscreenBtn.innerHTML = '<i class="fas fa-expand"></i>';
      }
      this.updateStatus('Exited fullscreen mode');
    }
  }

  toggleSound() {
    this.soundEnabled = !this.soundEnabled;
    if (this.soundToggleBtn) {
      this.soundToggleBtn.innerHTML = this.soundEnabled
        ? '<i class="fas fa-volume-up"></i>'
        : '<i class="fas fa-volume-mute"></i>';
    }
    this.updateStatus(`Sound ${this.soundEnabled ? 'enabled' : 'disabled'}`);
  }

  toggleTheme() {
    this.isDarkTheme = !this.isDarkTheme;
    if (this.isDarkTheme) {
      document.body.classList.add('shu-pdf-dark-theme');
      if (this.themeToggleBtn) {
        this.themeToggleBtn.innerHTML = '<i class="fas fa-sun"></i>';
      }
      if (this.themeText) {
        this.themeText.textContent = 'Dark';
      }
    } else {
      document.body.classList.remove('shu-pdf-dark-theme');
      if (this.themeToggleBtn) {
        this.themeToggleBtn.innerHTML = '<i class="fas fa-moon"></i>';
      }
      if (this.themeText) {
        this.themeText.textContent = 'Light';
      }
    }
    this.updateStatus(`${this.isDarkTheme ? 'Dark' : 'Light'} theme activated`);
  }
}

// Initialize the enhanced PDF viewer when the page loads
document.addEventListener('DOMContentLoaded', () => {
  setTimeout(() => {
    new EnhancedPDFViewer();
  }, 100);
});

// Prevent right-click context menu on canvas
document.addEventListener('contextmenu', (e) => {
  if (e.target.tagName === 'CANVAS') {
    e.preventDefault();
  }
});

// Prevent drag and drop on canvas
document.addEventListener('dragstart', (e) => {
  if (e.target.tagName === 'CANVAS') {
    e.preventDefault();
  }
});

</script>
